//
// spgasc
//
// This program reads data from a PRO-92 data file (SPG format or actual
// data core) and writes its contents in ASCII format.  That file may be
// user-edited, converted to SPG, and loaded back to the scanner.
//
// A SPG file is the format written by GRE's Data Manager.  There are three
// major sections:
//
// o A five byte header, hex values 02 00 00 00 00
// o Actual data, 26496 bytes
// o A trailing filler of 6272 byes, all hex 00
//
// This program deals with SPG files only.  Earlier versions worked with
// data core only files as well, but once released that feature was never
// used.  We now test for '.SPG' to protect the user.
//
// Written by Ken Plotkin
//            kjp15@cornell.edu
//            January 2001
//
// Translated from Fortran to C by Steve Falco
// 				   sfalco@worldnet.att.net
// 				   September 2001
// 				   This version is "Unix-centric" in that it
// 				   depends on various Unix system calls to
// 				   perform file I/O.  Debugged and tested on
// 				   RedHat Linux version 7.1.
//
// Distributed as careware.  If you like it, donate some money to a worthwhile
// charity.
//

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>

typedef unsigned char UCHAR;

#define SMALL 16	// Small buffer size
#define HLEN 5		// Header length
#define BLEN 26496	// Body length
#define EOL "\r\n"	// end of line

#define INFILE  "pro92.spg"
#define OUTFILE "pro92.asc"

void use();
void mot1(int ival, int isize, int *ib, int *ic);
void print_tag(FILE *fp, char *tp);

UCHAR *infile, *outfile;

main(int argc, char *argv[]) {

	int i, j, k, ii, ii2, ii3, jj, kk, b4;
	int ifreq, imode, infd, ibtype;
	int idpl, iradio, ireptr;
	int iarea, iskip, ival;
	int ia, ib, ic;
	int idval, isiz, itone, nchan;
	int ifleet[SMALL], ibyte3[SMALL];
	FILE *outfp;

	UCHAR *dot;
	UCHAR pldpl[SMALL], status[SMALL];
	UCHAR bytes[SMALL], clndat[BLEN+SMALL];
	UCHAR alpha[SMALL], alphas[SMALL], alphat[SMALL];
	UCHAR actbnk[SMALL];
	UCHAR trnkid[SMALL];
	UCHAR messag[SMALL][SMALL];

	double code, tonfrq;
	double stepf, freq, plfrq, frequ, freqhi, stepp;
	double freq0[] = { 29., 108., 137., 380., 806., 0., 0., 0. };
	double step[]  = { .0050, .0125, .0050, .0125, .0125, 0., 0., 0. };
	double pl0[]   = { 0, 0, 51.2,76.8,102.4,128.0,153.6,179.2,204.8,230.4};

	UCHAR cmode[][6] = {
		"AM   ","FM   ","PL   ","DPL  ","LTR  ","MOT  ", "EDACS"
	};
	UCHAR btype[][6] = {
		"Conv ","     ","     ","     ","LTR  ","MOT  ", "EDACS"
	};

	UCHAR bnkmod[][24] = {
		"Closed  00.0 kHz offset",
		"Open    00.0 kHz offset",
		"Closed  12.5 kHz offset",
		"Open    12.5 kHz offset",
		"Closed  25   kHz offset",
		"Open    25   kHz offset",
		"Closed  50   kHz offset",
		"Open    50   kHz offset"
	};

	while((ic = getopt(argc, argv, "h")) != -1) {
		switch(ic) {
			case 'h': /*FALLTHROUGH*/
			case '?':
				use();
				break;
		}
	}

	if(optind < argc) {
		infile = argv[optind++];
	} else {
		infile = INFILE;
	}

	if(optind < argc) {
		outfile = argv[optind++];
	} else {
		outfile = OUTFILE;
	}

	// Keep the user out of trouble - part 1
	if( !(					    // negate the whole thing
		((dot = strrchr(infile, '.')) != 0) // we found the last dot
		&&				    // and
		( 				    // (
		    (strcmp(dot, ".spg") == 0)	    // we have .spg
		    ||				    // or
		    (strcmp(dot, ".SPG") == 0)	    // we have .SPG
		)				    // )
	)) {
		fprintf(stderr, "Input file \"%s\" does not end in .spg or .SPG" EOL, infile);
		exit(1);
	}

	// Keep the user out of trouble - part 2
	// This won't catch every case, but it might help
	if(strcmp(infile, outfile) == 0) {
		fprintf(stderr, "Input file \"%s\" is the same as output file \"%s\"" EOL, infile, outfile);
		exit(1);
	}

	if((infd = open(infile, O_RDONLY)) == -1) {
		fprintf(stderr, "Cannot open %s" EOL, infile);
		exit(1);
	}

	// Trim off the first 5 bytes.
	if(read(infd, bytes, HLEN) != HLEN) {
		fprintf(stderr, "Cannot read header of %s" EOL, infile);
		exit(1);
	}

	// Load the rest of the file (make it look like array is 1-based)
	if(read(infd, clndat + 1, BLEN) != BLEN) {
		fprintf(stderr, "Cannot read body of %s" EOL, infile);
		exit(1);
	}

	close(infd);

	if((outfp = fopen(outfile, "w")) == NULL) {
		fprintf(stderr, "Cannot create/open %s" EOL, outfile);
		exit(1);
	}

	fprintf(outfp, "SPGASC Version 1.03, Verbose format" EOL);
	fprintf(outfp, "Edit this file per the instructions.  Spacing matters!" EOL EOL);

	// When picking through the data, JJ is the pointer to the field in
	// the data file, counting from 0.  That makes it easy to figure when
	// browsing a hex dump.  The data array here begins at 1, so we take
	// data beginning at JJ+1.  That makes loops easy, since a 12-byte
	// item will be JJ plus 1 to 12.

	// Banks each occupy 2443 bytes.  The first 900 bytes are the channels,
	// 18 bytes each.  This is followed by 1543 bytes of search, trunk,
	// etc., data

	// Weather channels appear after the last bank, in a pretty similar
	// format.  These are followed by preprogrammed search bands, also
	// pretty similar.  We have kept those in the sequence as if they were
	// an eleventh bank.  There are some format differences, which are
	// accounted for by alternate code as necessary.  In retrospect, I
	// wish I had kept them separate, rather than have all those IFs.
	// The next line (nchan=50) is one of those details: it's the number
	// of channels per bank, and has to be changed to something else for
	// the weather/search eleventh bank.

	nchan = 50;

	for(k = 0; k <= 10; k++) {
		if(k <= 9) {
			for(i = 1; i <= 12; i++) {
				alphas[i] = clndat[k*2443+906+i];
				alphat[i] = clndat[k*2443+921+i];
			}
			alphas[i] = 0; //null terminate
			alphat[i] = 0; // null terminate

			jj = k*2443 + 933;
			ii = clndat[jj+1];

			jj = k*2443 + 934;
			ibtype = clndat[jj+1];

			fprintf(outfp, "****** Bank%2d ******   Tag: %s  Type: %s  Mode: %s" EOL, k, alphat+1, btype[ibtype], (ii & 0x2) ? "Open  " : "Closed");
		} else {
			fprintf(outfp, "****** Weather Channels ******" EOL);
		}
		fprintf(outfp, " " EOL);

		// For "Bank 10", we have 7 weather plus 3 unused plus the
		// number of preprogrammed search bands
		if(k == 10) {
			nchan = clndat[0x6769+1] + 10;
		}

		for(j = 1; j <= nchan; j++) {
			if(k == 10 && j == 11) {
				fprintf(outfp, EOL "****** Preprogrammed Search Bands ******" EOL EOL);
			}

			// Channels are in 18 byte blocks.  The first six are
			// data.  The remaining 12 are the alpha tag.
			jj = (k * 2443) + ((j-1) * 18);
			for(ii = 1; ii <= 6; ii++) {
				bytes[ii] = clndat[jj+ii];
			}
			for(ii = 1; ii <= 12; ii++) {
				alpha[ii] = clndat[jj+6+ii];
			}

			// Frequency range is high nybble of third byte.
			// Mode is in the low nybble.
			ifreq = (bytes[3] >> 4) & 0x07;
			imode = bytes[3] & 0x07;

			// In Search Bank, range and mode are in fourth byte
			// and the nybbles are swapped.

			// Low nybble of Search Band Byte 3 gives the search
			// step size, in terms of the number of frequency
			// steps.
			if((k == 10) && (j >= 11)) {
				imode = (bytes[4] >> 4) & 0x07;
				ifreq = bytes[4] & 0x07;
				stepf = step[ifreq] * (double)bytes[3];
			}

			// Frequency is base for range plus first byte plus
			// second byte times step
			freq = freq0[ifreq] + (double)bytes[1]
				+ (double)bytes[2] * step[ifreq];

			// If ifreq or imode is out of range, the channel is
			// vacant - SAF 2001-10-01
			if(ifreq > 4 || imode > 6) {
				freq = 0.0;
			}

			// If bit 4 of byte 4 is set, the channel is vacant
			if((bytes[4] & 8) == 8) {
				freq = 0.0;
			}

			// PL and DPL for Banks 0-9; "Bank" 10 is different.
			// PL tone: range, as defined by Byte 6, plus 0.1
			// times byte 5.  DPL tone is D, plus three digit
			// octal value of bytes 5 & 6 (6 is high)
			//
			// Default for no tone
			strcpy(pldpl, "     0.0");
			if(k <= 9) {
				if((imode == 2) &&
				   (bytes[6] >= 2) && 
				   (bytes[6] <= 9)) {
					plfrq = pl0[bytes[6]] +
					  ((double)bytes[5] * 0.1);
					sprintf(pldpl, "%8.1f", plfrq);
				}

				if(imode == 3) {
					strcpy(pldpl, "    D   ");
					idpl = (bytes[6] << 8) | bytes[5];
					sprintf(pldpl + 5, "%3.3o", idpl);
				}
			}

			// "Bank" 10 search banks use Bytes 5 and 6 to define
			// the upper limit of each bank.  Role of Byte 3
			// (frequency range and mode) is as for channels,
			// except the nybbles are swapped (it's mode and
			// range.)
			if((k == 10) && (j >= 11) && (freq > 0.0)) {
				frequ = freq0[ifreq] + (double)bytes[5] +
					((double)bytes[6] * step[ifreq]);
			} else {
				frequ = 0.0;
			}

			// Set up the string showing delay, atten, scan status,
			strcpy(status, "----");
			if(!(k == 10 && j >= 11)) {
				if((bytes[4] & 8) == 8) status[0] = 'U';
				if((bytes[4] & 4) == 4) status[1] = 'D';
				if((bytes[4] & 2) == 2) status[2] = 'A';
				if((bytes[4] & 1) == 1) status[3] = 'L';
			}

			// Write it out.  Search bands use a different format
			// than channels.
			if((k == 10) && (j >= 11)) {
				fprintf(outfp, "%3d  ", j - 11);
				for(ii = 1; ii <= 6; ii++) {
					fprintf(outfp, "%02X ", bytes[ii]);
				}
				print_tag(outfp, alpha);
				fprintf(outfp, "%10.4f  %5s%8.4f%8.4f" EOL,
					freq, cmode[imode], frequ, stepf);
			} else {
				fprintf(outfp, "%3d  ", j - 1);
				for(ii = 1; ii <= 6; ii++) {
					fprintf(outfp, "%02X ", bytes[ii]);
				}
				print_tag(outfp, alpha);
				fprintf(outfp, "%10.4f  %5s%8s  %4s" EOL,
					freq, cmode[imode], pldpl, status);
			}
		}
		fprintf(outfp, " " EOL);
		
		// The 900 bytes of channel data are followed by bank search
		// info, bank alpha tag and trunking info, search lockouts,
		// and talkgroup IDs.
		if(k == 10) continue;

		// Bank search data.  Six data bytes (similar to channels),
		// alpha tag, then three more data bytes
		jj = k*2443 + 900;

		for(ii = 1; ii <= 6; ii++) {
			bytes[ii] = clndat[jj+ii];
		}
		for(ii = 1; ii <= 12; ii++) {
			alpha[ii] = clndat[jj+6+ii];
		}
		for(ii = 1; ii <= 3; ii++) {
			ibyte3[ii] = clndat[jj+18+ii];
		}

		// The first six bytes of bank search are interpreted exactly
		// as in Channel data
		ifreq = (bytes[3] >> 4) & 0x07;
		imode = bytes[3] & 0x07;

		freq = freq0[ifreq] + (double)bytes[1] +
			(double)bytes[2] * step[ifreq];

		// If ifreq or imode is out of range, the channel is
		// vacant - SAF 2001-10-01
		if(ifreq > 4 || imode > 6) {
			freq = 0.0;
		}

		strcpy(status, "----");
		if(!(k == 10 && j >= 10)) {
			if((bytes[4] & 8) == 8) status[0] = 'U';
			if((bytes[4] & 4) == 4) status[1] = 'D';
			if((bytes[4] & 2) == 2) status[2] = 'A';
			if((bytes[4] & 1) == 1) status[3] = 'L';
		}

		// PL/DPL, as for channels.
		strcpy(pldpl, "     0.0");
		if(k <= 9) {
			if((imode == 2) &&
			   (bytes[6] >= 2) &&
			   (bytes[6] <= 9)) {
				plfrq = pl0[bytes[6]] + (double)bytes[5]*0.1;
				fprintf(outfp, "%8.1f" EOL, plfrq);
			}
			if(imode == 3) { 
				strcpy(pldpl, "    D   ");
				fprintf(outfp, "%1x%02x" EOL, bytes[6],bytes[5]);
			}
		}

		// Upper limit of search range is given by first two bytes of
		// 3-byte set.  Search step is given by third byte of that set
		freqhi = freq0[ifreq] + (double)ibyte3[1] +
			(double)ibyte3[2] * step[ifreq];
		stepp = step[ifreq] * (double)ibyte3[3];

		fprintf(outfp, "Srch ");
		for(ii = 1; ii <= 6; ii++) {
			fprintf(outfp, "%02X ", bytes[ii]);
		}
		print_tag(outfp, alpha);
		fprintf(outfp, "%10.4f  %s%s  %s" EOL,
			freq, cmode[imode], pldpl, status);
		fprintf(outfp, "End, step     ");
		for(ii = 1; ii <= 3; ii++) {
			fprintf(outfp, "%02X ", ibyte3[ii]);
		}
		fprintf(outfp, "               %10.4f%8.4f" EOL,
			freqhi, stepp);
		fprintf(outfp, " " EOL);

		// Scan bank info
		fprintf(outfp, " Bank%2d info:" EOL, k);
		jj = k*2443 + 921;

		// 12 character alpha tag
		for(ii = 1; ii <= 12; ii++) {
			alpha[ii] = clndat[jj+ii];
		}
		alpha[ii] = 0; // null terminate
		fprintf(outfp, "%s              ", alpha+1);
		print_tag(outfp, alpha);
		fprintf(outfp, "" EOL);

		// Bank mode and Motorola trunking offset
		jj = k*2443 + 933;
		ii = clndat[jj+1];
		fprintf(outfp, " %02X                         %s          (Mode/Offset)" EOL,
			ii, bnkmod[ii/2]);

		// Bank type
		jj = k*2443 + 934;
		ibtype = clndat[jj+1];
		fprintf(outfp, " %02X                         %s                            (Bank type)" EOL,
			ibtype, btype[ibtype]);

		// Fleet map
		jj = k*2443 + 935;
		for(ii = 1; ii <= 8; ii++) {
			fprintf(outfp, " %02X", clndat[jj+ii]);
		}
		fprintf(outfp, "    ");
		for(ii = 1; ii <= 8; ii++) {
			fprintf(outfp, "S%02d ", clndat[jj+ii]);
		}
		fprintf(outfp, " (Fleet map)" EOL);
		for(ii = 0; ii <= 7; ii++) {
			ifleet[ii] = clndat[jj+1+ii];
		}
		fprintf(outfp, " " EOL);

		// Search lockouts
		fprintf(outfp, " Search lockouts:" EOL);
		for(kk = 1; kk <= 50; kk++) {
			jj = k*2443 + 943 + (kk-1)*2;
			bytes[1] = clndat[jj+1];
			bytes[2] = clndat[jj+2];
			if((bytes[1] != 255) && (bytes[2] != 255)) {
				freq = freq0[ifreq] + (double)bytes[1] +
			           (double)bytes[2] * step[ifreq];
				fprintf(outfp, "%3d %02X %02X%10.4f" EOL,
					kk, bytes[1], bytes[2], freq);
			}
		}
		fprintf(outfp, " (end lockouts)" EOL);
		fprintf(outfp, " " EOL);

		// Talkgroup IDs.  Always write 100 fields, regardless of
		// whether any IDs are used.

		// There are four nybbles, the first of which is Scan/Skip and
		// the rest of which are the ID.  If an ID slot is unused,
		// display a blank.  If it's unused but the Skip nybble is
		// set then display Skip.
		fprintf(outfp, " Talkgroup IDs:" EOL);
		for(kk = 1; kk <= 100; kk++) {
			sprintf(trnkid, "           ");
			jj = k*2443 + 1043 + (kk-1)*14;
			bytes[1] = clndat[jj+1];
			bytes[2] = clndat[jj+2];

			// LTR ID
			if(ibtype == 4) {
				iradio = bytes[1];
				ireptr = bytes[2] & 31;
				iarea  = (bytes[2] & 32)/32;
				iskip  = (bytes[2] & 128)/128;
				sprintf(trnkid, "%06d %s",
					iradio + (1000*ireptr) + (100000*iarea),
					(iskip == 1) ? "Skip" : "Scan");

				// 000000 is no ID.  But allow "Skip" to stand
				// if that's set
				if(     !strcmp(trnkid, "000000 Skip")) {
					sprintf(trnkid, "       Skip");
				}
				if(     !strcmp(trnkid, "000000 Scan")) {
					sprintf(trnkid, "           ");
				}
			}

			// Motorola trunking. Assemble the two bytes
			// into a 16 bit value, then extract the bank
			// and check against the fleet map.  If the
			// size is greater than 0, it's Type 1 and we
			// use routine mot1 to extract the fleet and
			// sublfleet from the value.
			if(ibtype == 5) {
				ival = bytes[2]*256 + bytes[1];
				ia = (ival & 0xe00) / 0x200;
				isiz = ifleet[ia];
			}

			if(ibtype == 5  &&  isiz > 0) {
				mot1(ival, isiz, &ib, &ic);

				// Block size for S12-14 are rounded
				// down to even, fours, eights
				if(isiz == 12) ia = (ival & 0xc00)/0x200;
				if(isiz == 13) ia = (ival & 0x800)/0x200;
				if(isiz == 14) ia = 0;

				if(isiz == 1) {
					sprintf(trnkid, "%1.1d%3.3d-%1.1d",
						ia, ib, ic);
				} else {
					sprintf(trnkid, "%1.1d%2.2d-%2.2d",
						ia, ib, ic);
				}

				// Skip/Scan if high nybble is 8 or 0,
				// respectively
				if((ival & 0x8000) ==  0) {
					strcpy(trnkid + 6, " Scan");
				} else {
					strcpy(trnkid + 6, " Skip");
				}

				// Update 1-27-01: 0000 is always no ID
				if(ival == 0) {
					sprintf(trnkid, "           ");
				}

				// IVAL of 8000 is also no ID, but
				// "Skip" stays
				if(ival == 0x8000) {
					sprintf(trnkid, "       Skip");
				}
			}

			// Motorola Type 2 ID (Or S00 in a hybrid
			// system).
			if((ibtype == 5) && (isiz == 0)) {
				b4 = (bytes[2] << 8) | bytes[1];
				idval = b4 & 0x0fff;
				sprintf(trnkid, "%05d %s",
					idval*16,
					((b4 & 0xf000) == 0x8000) ?
					"Skip " : "Scan " );

				// 1-27-01 - Either 8000 or 0000 are
				// no ID, but "Skip" stays if 8000
				if(     !strcmp(trnkid, "00000 Skip ")) {
					sprintf(trnkid, "      Skip ");
				}
				if(     !strcmp(trnkid, "00000 Scan ")) {
					sprintf(trnkid, "           ");
				}
			}

			// EDACS ID
			if(ibtype == 6) {
				b4 = (bytes[2] << 8) | bytes[1];
				idval = b4 & 0x0fff;
				sprintf(trnkid, "%4d %s",
					idval,
					((b4 & 0xf000) == 0x8000) ?
					"Skip  " : "Scan  " );

				// 1-27-01  8000 or 0000 are no ID, but
				// "Skip" stays for 8000
				if(b4 == 0x8000) {
					sprintf(trnkid, "     Skip  ");
				}
				if(b4 == 0x0000) {
					sprintf(trnkid, "           ");
				}
			}

			for(ii = 1; ii <= 12; ii++) {
				alpha[ii] = clndat[jj+2+ii];
			}
			fprintf(outfp, "%4d %02X%02X     [%s]   ", 
				kk, bytes[2], bytes[1], trnkid);
			print_tag(outfp, alpha);
			fprintf(outfp, "" EOL);
		}
		fprintf(outfp, " (end talk IDs)" EOL);
		fprintf(outfp, " " EOL);
	}

	// Message
	fprintf(outfp, "****** Opening screen message ******" EOL);
	fprintf(outfp, " " EOL);
	for(i = 1; i <= 4; i++) {
		jj = 26410 + (i-1)*12;
		for(j = 1; j <= 12; j++) {
			messag[i][j] = clndat[jj+j];
		}
		messag[i][j] = 0; // null terminate
		fprintf(outfp, " >%s<" EOL, messag[i]+1);
	}
	fprintf(outfp, " " EOL);
	fprintf(outfp, "****** Scanner settings ******" EOL);
	fprintf(outfp, " " EOL);

	ii = clndat[0x675a + 1];
	fprintf(outfp, " %02X                 %02X   Unknown.  Initialized 14, default 00 (hex)" EOL, ii, ii);

	ii = clndat[0x675b + 1];
	fprintf(outfp, " %02X         %10d   Backlight on time, 1 sec to 255 sec" EOL, ii, ii);

	ii = clndat[0x675c + 1];
	fprintf(outfp, " %02X         %10d   Scan delay time, 400 msec to 25500 msec" EOL, ii, ii*100);

	ii = clndat[0x675d + 1];
	fprintf(outfp, " %02X         %10d   Trunk rescan delay time,400 msec to 10000 msec" EOL, ii, ii*100);

	ii = clndat[0x675e + 1];
	fprintf(outfp, " %02X                 %02X   Unknown.  Usually 2.55 times trunk delay byte" EOL, ii, ii);

	ii = clndat[0x675f + 1];
	fprintf(outfp, " %02X%19.1f   Minimum scan delay time, 0.4 sec to 25.5 sec" EOL, ii, (double)ii*.1);

	ii = clndat[0x6760 + 1];
	fprintf(outfp, " %02X              %5i   Display contrast, 09 (light) to 14 (blacked out)" EOL, ii, ii);

	ii  = clndat[0x6761 + 1];
	ii2 = clndat[0x6762 + 1];
	fprintf(outfp, " %02X %02X           %2d%3.2d   Priority channel number" EOL, ii, ii2, ii2, ii);

	ii  = clndat[0x6763 + 1];
	ii2 = clndat[0x6764 + 1];
	ii3 = clndat[0x6765 + 1];
	fprintf(outfp, " %02X %02X %02X     %02X %02X %02X   Last search frequency.  Initialized 00 50 00" EOL,
		ii, ii2, ii3, ii, ii2, ii3);

	ii = clndat[0x6766 + 1];
	fprintf(outfp, " %02X                 %02X   Priority: 01 = on, 00 = off" EOL, ii, ii);

	ii = clndat[0x6767 + 1];
	fprintf(outfp, " %02X                 %02X   Unknown byte.  Initialized 00" EOL, ii, ii);

	ii = clndat[0x6768 + 1];
	fprintf(outfp, " %02X                 %02X   Unknown byte.  Initialized 00" EOL, ii, ii);

	ii = clndat[0x6769 + 1];
	fprintf(outfp, " %02X%19d   Number of preprogrammed search bands (100 max)" EOL, ii, ii);

	ii  = clndat[0x676a + 1];
	ii2 = clndat[0x676b + 1];
	code = (double)(ii2*256 + ii);
	tonfrq = 1000000.0 / (65536.0 - code);
	itone = tonfrq + 0.3;
	fprintf(outfp, " %02X %02X          %6d   Keypad tone frequency, 300 Hz to 3000 Hz" EOL, ii, ii2, itone);

	for(i = 1; i <= 10; i++) {
		actbnk[i] = clndat[0x676c+i];
		if((actbnk[i] < '0') || (actbnk[i] > '9')) {
			actbnk[i] = '-';
		}
	}
	actbnk[i] = 0; // null terminated

	for(ii = 1; ii <= 10; ii++) {
		fprintf(outfp, "%c", clndat[0x676c+ii]);
	}
	fprintf(outfp, "  %s   Enabled scan banks" EOL, actbnk+1);

	for(i = 1; i <= 10; i++) {
		actbnk[i] = clndat[0x6776+i];
		if((actbnk[i] < '0') || (actbnk[i] > '9')) {
			actbnk[i] = '-';
		}
	}
	actbnk[i] = 0; // null terminated

	for(ii = 1; ii <= 10; ii++) {
		fprintf(outfp, "%c", clndat[0x6776+ii]);
	}
	fprintf(outfp, "  %s   Active search banks" EOL, actbnk+1);

	exit(0);
}

void
mot1(int ival, int isize, int *ib, int *ic) {

	// This routine extracts the fleet (ib) and subfleet(ic) from code
	// ival, for fleet size isize.  Block (which would be ia) is not
	// extracted here; it had to have been extracted prior to calling
	// this routine so that the appropriate value of isize could be
	// determined from the fleet map.

	// All ISIZEs are covered
	int nsub, incr, jsubc, maskc, maskb, idivc, idivb;
	static int nsubs[] = {  0,  4,  8,  8, 16,  4,  8,  4,
				4,  4,  8, 16, 16, 16, 16,  0 };
	static int incrs[] = {  0,  1,  4,  8, 32,  2,  2,  4,
				8, 16, 16, 16, 64,128,256,  0 };

	nsub = nsubs[isize & 0x0f];
	incr = incrs[isize & 0x0f];

	jsubc = incr-1;
	maskc = incr*nsub-1;
	maskb = 0x1ff - maskc;
	idivc = incr;
	idivb = incr*nsub;

	if(isize <= 11) {
	   *ib = (ival & maskb) / idivb;
	} else {
	   *ib = 0;
	}
	*ic = ((ival-jsubc) & maskc) / idivc;
	return;
}

void
print_tag(FILE *fp, char *tp)
{
	int ii;

	fprintf(fp, " >");
	for(ii = 1; ii <= 12; ii++) {
		fprintf(fp, "%c", tp[ii]);
	}
	fprintf(fp, "<");
}

void
use()
{
	fprintf(stderr, "spgasc [-h] [inputfile.spg] [outputfile]" EOL);
	fprintf(stderr, "  -h prints this help message" EOL);
	fprintf(stderr, "  inputfile.spg is the binary input file" EOL);
	fprintf(stderr, "          (defaults to %s, name must end with .spg)" EOL, INFILE);
	fprintf(stderr, "  outputfile is the ascii output file" EOL);
	fprintf(stderr, "          (defaults to %s)" EOL, OUTFILE);
}
